/* editor.js (修正版) */
class SyntaxEditor {
    /**
     * @param {HTMLTextAreaElement} textareaElement - エディタ化する対象のtextarea要素
     */
    constructor(textareaElement) {
        this.textarea = textareaElement;

        if (!this.textarea || this.textarea.tagName !== 'TEXTAREA') {
            console.error('Invalid element provided. Must be a TEXTAREA.');
            return;
        }

        this.setupDOM(); // DOM構造の自動生成
        this.bindEvents();
        this.updateHighlight(); // 初期描画
    }

    /**
     * エディタに必要なコンテナとハイライトレイヤーを自動生成し、DOMに挿入する
     */
    setupDOM() {
        const originalParent = this.textarea.parentNode;

        // 1. 全体コンテナの作成
        this.container = document.createElement('div');
        this.container.className = 'editor-container';

        // 2. ラッパー要素の作成（position: relative の基準点となる）
        const wrapper = document.createElement('div');
        wrapper.className = 'editor-wrapper';

        // 3. ハイライトレイヤーの作成
        this.highlightLayer = document.createElement('div');
        this.highlightLayer.className = 'highlight-layer editor-font-base';

        // 4. 元のtextareaをコンテナとラッパーでラップする
        originalParent.insertBefore(this.container, this.textarea);
        this.container.appendChild(wrapper);

        // 5. クラスを適用し、構造を組み立てる
        this.textarea.classList.add('editor-textarea', 'editor-font-base');
        this.textarea.style.resize = 'none';

        // textarea を wrapper に追加
        wrapper.appendChild(this.textarea);

        // **変更点:** textarea の直後にハイライトレイヤーを兄弟要素として挿入
        // 実際は wrapper 内で兄弟関係になる
        wrapper.insertBefore(this.highlightLayer, this.textarea.nextSibling);

        // **重要**: ハイライトレイヤーがtextareaの背面に表示されるように、
        // z-indexを調整するか、挿入順序を調整する必要があります。
        // 現在のCSS設計（highlight-layerに pointer-events: none; がある前提）では、
        // 挿入順序に関わらず、CSSの position: absolute; で両者が重なり、
        // z-indexによって表示順序が決まります。
    }

    // 以下の applyHighlight, updateHighlight, bindEvents, initializeAll, static initializeAll の部分は変更なし

    /**
     * 文字列をエスケープし、ハイライト用のタグを埋め込む
     */
    applyHighlight(text) {
        let escaped = text
            .replace(/&/g, '&amp;')
            .replace(/</g, '&lt;')
            .replace(/>/g, '&gt;');

        escaped = escaped.replace(/(&lt;.*?&gt;)/g, '<span class="angle-bracket">$1</span>');
        escaped = escaped.replace(/(\{.*?\})/g, '<span class="curly-bracket">$1</span>');

        return escaped;
    }

    /**
     * ハイライトレイヤーの更新
     */
    updateHighlight() {
        const text = this.textarea.value;
        const highlighted = this.applyHighlight(text);
        this.highlightLayer.innerHTML = highlighted;
    }

    /**
     * イベントリスナーの設定
     */
    bindEvents() {
        this.textarea.addEventListener('input', () => this.updateHighlight());
        this.textarea.addEventListener('scroll', () => {
            this.highlightLayer.scrollTop = this.textarea.scrollTop;
            this.highlightLayer.scrollLeft = this.textarea.scrollLeft;
        });
    }

    /**
     * ページ内のすべてのエディタ化対象要素を探して初期化する静的メソッド
     */
    static initializeAll() {
        document.querySelectorAll('textarea[data-syntax-editor]').forEach(textarea => {
            if (!textarea.dataset.initialized) {
                new SyntaxEditor(textarea);
                textarea.dataset.initialized = 'true';
            }
        });
    }
}

document.addEventListener('DOMContentLoaded', SyntaxEditor.initializeAll);