/**
 * Console2 List JavaScript
 * 記事一覧とメッセージ一覧で共通のスクリプト
 */

var ConsoleList = (function () {
    'use strict';

    // プライベート変数
    var searchInput = null;
    var searchClear = null;
    var sortType = null;
    var checkboxes = null;
    var selectedCountEl = null;
    var deleteBtn = null;

    // 設定
    var config = {
        deletePath: '',      // 例: '/api/article/delete' or '/api/message/delete' or '/api/v2/messages'
        listPath: '',         // 例: '/console/articles' or '/console/messages'
        checkboxSelector: '.article-checkbox, .message-checkbox',
        deleteConfirmMessage: '件を削除しますか？この操作は取り消せません。',
    };

    /**
     * 検索を実行
     */
    function doSearch() {
        var q = searchInput ? searchInput.value.trim() : '';
        var sort = sortType ? sortType.value : '0';
        var url = BASE_URL + config.listPath + '?page=1&sort_type=' + sort;
        if (q) {
            url += '&q=' + encodeURIComponent(q);
        }
        location.href = url;
    }

    /**
     * 選択状態を更新
     */
    function updateSelection() {
        var checked = document.querySelectorAll(config.checkboxSelector + ':checked');
        if (selectedCountEl) {
            selectedCountEl.textContent = checked.length;
        }
        if (deleteBtn) {
            deleteBtn.disabled = checked.length === 0;
        }
    }

    /**
     * 選択したアイテムを削除
     */
    function deleteSelected() {
        var checked = document.querySelectorAll(config.checkboxSelector + ':checked');
        if (checked.length === 0) return;

        var ids = Array.from(checked).map(function (cb) {
            return cb.getAttribute('data-id');
        });

        if (confirm(ids.length + config.deleteConfirmMessage)) {
            // 削除API呼び出し
            var promises = ids.map(function (id) {
                // RESTful API形式: DELETE /api/v2/messages/{id}
                return fetch(BASE_URL + config.deletePath + '/' + id, {
                    method: 'DELETE',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.getElementById('csrf_token').value
                    }
                });
            });

            Promise.all(promises).then(function () {
                location.reload();
            }).catch(function (err) {
                alert('削除中にエラーが発生しました');
                console.error(err);
            });
        }
    }

    /**
     * タグクリックイベントを設定
     */
    function setupTagClick() {
        document.querySelectorAll('.p-tag').forEach(function (tag) {
            tag.addEventListener('click', function () {
                var search = this.getAttribute('data-search') || this.textContent.trim();
                if (searchInput) {
                    searchInput.value = search;
                    doSearch();
                }
            });
        });
    }

    /**
     * イベントリスナーを設定
     */
    function setupEventListeners() {


        // 検索入力
        if (searchInput) {
            searchInput.addEventListener('keypress', function (e) {
                if (e.key === 'Enter') {
                    doSearch();
                }
            });
        }

        // 検索クリア
        if (searchClear) {
            searchClear.addEventListener('click', function () {
                if (searchInput) {
                    searchInput.value = '';
                    doSearch();
                }
            });
        }

        // ソート変更
        if (sortType) {
            sortType.addEventListener('change', function () {
                doSearch();
            });
        }

        // チェックボックス変更
        if (checkboxes) {
            checkboxes.forEach(function (cb) {
                cb.addEventListener('change', updateSelection);
            });
        }

        // 削除ボタン
        if (deleteBtn) {
            deleteBtn.addEventListener('click', deleteSelected);
        }

        // タグクリック
        setupTagClick();
    }

    /**
     * 初期化
     * @param {Object} options - 設定オプション
     */
    function init(options) {
        // 設定をマージ
        if (options) {
            if (options.deletePath) config.deletePath = options.deletePath;
            if (options.listPath) config.listPath = options.listPath;
            if (options.checkboxSelector) config.checkboxSelector = options.checkboxSelector;
            if (options.deleteConfirmMessage) config.deleteConfirmMessage = options.deleteConfirmMessage;
            if (options.useRestfulApi !== undefined) config.useRestfulApi = options.useRestfulApi;
        }

        // 要素の取得
        searchInput = document.getElementById('searchInput');
        searchClear = document.getElementById('searchClear');
        sortType = document.getElementById('sortType');
        checkboxes = document.querySelectorAll(config.checkboxSelector);
        selectedCountEl = document.getElementById('selectedCount');
        deleteBtn = document.getElementById('deleteSelected');

        // イベントリスナーを設定
        setupEventListeners();
    }

    // 公開API
    return {
        init: init,
        doSearch: doSearch,
        updateSelection: updateSelection,
        deleteSelected: deleteSelected
    };
})();
