<?php

namespace pictpostpersonal;

/**
 * コントローラ(アクション)の実行結果をまとめたクラス (PHP 7.1対応版)
 */
class Result
{
    /** @var string|null 出力内容 (HTMLなど) */
    public $output = null;

    /** @var int HTTPステータスコード (200,301,307,404, etc) */
    public $status = 200;

    /** @var string|null リダイレクト先URL */
    public $redirect = null;

    /**
     * @var mixed リダイレクト先のアクション(配列やコールバック)
     *            PHP 7.1では union types が使えないため、ここでは mixed
     */
    public $redirect_action = null;

    /** @var string|null MIMEタイプ (text/html, application/jsonなど) */
    public $mime_type = null;

    /** @var bool 次の処理を続行するか (true:継続, false:終了) */
    public $next = true;

    /** @var mixed リクエストを更新したい場合に差し替えるオブジェクト */
    public $updated_request = null;

    /**
     * 即時リダイレクト
     *
     * @param string $url
     * @param int    $redirectCode (301, 307など)
     * @return void
     */
    public static function redirect($url, $redirect_code = -1)
    {
        if ($redirect_code === -1) {
            header("Location: $url");
        } else {
            header("Location: $url", true, $redirect_code);
        }
        exit;
    }

    /**
     * コンストラクタ
     *
     * @param int    $code         HTTPステータスコード
     * @param string $output       出力内容
     * @param string $redirect_url リダイレクト先URL
     * @param mixed  $redirect_action 別アクション
     * @param string $mime_type
     * @param bool   $next
     */
    public function __construct(
        $code = 200,
        $output = '',
        $redirect_url = '',
        $redirect_action = null,
        $mime_type = 'text/html',
        $next = true
    ) {
        $this->status   = $code;
        $this->next     = $next;
        $this->redirect = null;
        $this->redirect_action = null;
        $this->mime_type = null;
        $this->output = null;

        // redirect_action があれば最優先
        if (!empty($redirect_action)) {
            $this->redirect_action = $redirect_action;
            $this->output    = '';
            return;
        }

        // 3xxの範囲ならリダイレクトURLをセット
        if ($code >= 300 && $code <= 399 && $redirect_url !== '') {
            $this->redirect = $redirect_url;
            $this->output   = '';
            return;
        }

        // 通常出力(200や404など)
        $this->output    = $output;
        $this->mime_type = $mime_type;
    }
}
