<?php

namespace pictpostpersonal;

/**
 * ログ出力用クラス (PHP 7.1対応版)
 */
class Logger
{
    /** @var string ログファイルパス */
    private $log_file;

    /** @var int ログレベル閾値 */
    private $log_level;

    // ログレベル定数
    public const LOG_LEVEL_ALL      = 0;
    public const LOG_LEVEL_NORMAL   = 10;
    public const LOG_LEVEL_WARN     = 50;
    public const LOG_LEVEL_ERROR    = 100;
    public const LOG_LEVEL_SECURITY = 900;
    public const LOG_LEVEL_CRITICAL = 900;

    /**
     * コンストラクタ
     *
     * @param string $log_file  ログ出力先ファイル
     * @param int    $log_level このレベル以上のログを記録
     */
    public function __construct($log_file, $log_level = self::LOG_LEVEL_WARN)
    {
        $this->log_file  = $log_file;
        $this->log_level = $log_level;
    }

    /**
     * ログを書き込む
     *
     * @param string $message
     * @param int    $log_level
     */
    public function log($message, $log_level = self::LOG_LEVEL_WARN)
    {
        if ($log_level >= $this->log_level) {
            $timestamp = date('Y-m-d H:i:s');
            $log_message = "[$timestamp] $message" . PHP_EOL;
            file_put_contents($this->log_file, $log_message, FILE_APPEND | LOCK_EX);
        }
    }

    /**
     * ログを書き込み + 画面にも表示
     *
     * @param string $message
     * @param int    $log_level
     */
    public function logWithShow($message, $log_level = self::LOG_LEVEL_WARN)
    {
        if ($log_level >= $this->log_level) {
            $timestamp = date('Y-m-d H:i:s');
            $log_message = "[$timestamp] $message" . PHP_EOL;
            file_put_contents($this->log_file, $log_message, FILE_APPEND | LOCK_EX);
            echo str_replace("\r\n", '<br>', $message);
        }
    }

    /**
     * ログレベルを変更
     *
     * @param int $log_level
     */
    public function setLogLevel($log_level)
    {
        $this->log_level = $log_level;
    }
}
