<?php

namespace pictpostpersonal;

class File
{
    public $name;
    public $tmp_name;
    public $error = 0;
    public $type = '';
    const HAS_DOUBLE_EXTENSION = 900;
    const NO_UPLOAD_FILE = 901;

    function __construct(array $file)
    {
        $this->name = $file['name'];
        $this->tmp_name = $file['tmp_name'];
        $this->error = $file['error'];
        $this->type = $file['type'];
        if ($this->name === '') {
            $this->error = 0;
        }
        if ($this->error == 0 && $this->hasDoubleExtension($this->name)) {
            $this->error = self::HAS_DOUBLE_EXTENSION;
        }
        if (!is_uploaded_file($this->tmp_name)) {
            $this->error = self::NO_UPLOAD_FILE;
        }
    }

    public function getError(): int
    {
        return $this->error;
    }
    public function getErrorMessage(): string
    {
        if ($this->error !== UPLOAD_ERR_OK) {
            switch ($this->error) {
                case UPLOAD_ERR_INI_SIZE:
                    return 'ファイルが大きすぎます';
                case UPLOAD_ERR_FORM_SIZE:
                    return 'ファイルが大きすぎます';
                case UPLOAD_ERR_PARTIAL:
                    return 'ファイルの一部のみアップロードされました';
                case UPLOAD_ERR_NO_FILE:
                    return 'ファイルがアップロードされていません';
                case self::HAS_DOUBLE_EXTENSION:
                    return '二重拡張子は許可されていません';
                case self::NO_UPLOAD_FILE:
                    return 'アップロードエラーが発生しました';
                default:
                    return 'アップロードエラーが発生しました';
            }
        }
        return '';
    }

    public function moveTo($path)
    {
        move_uploaded_file($this->tmp_name, $path);
    }

    public function getMediaType()
    {
        $finfo = \finfo_open(FILEINFO_MIME_TYPE);
        $mime_type = \finfo_file($finfo, $this->tmp_name);

        if (strpos($mime_type, 'video/') === 0) {
            return 'movie';
        }
        if (strpos($mime_type, 'image/') === 0) {
            return 'image';
        }
        return 'unknown';
    }

    public function checkAllowedExtension(array $allowed_extensions): bool
    {
        $extension = $this->getExtension();
        if (!in_array($extension, $allowed_extensions)) {
            return false;
        }
        return true;
    }
    public function getExtension(): string
    {
        return strtolower(pathinfo($this->name, PATHINFO_EXTENSION));
    }
    function hasDoubleExtension(string $filename): bool
    {
        $filename = strtolower($filename);

        // 拡張子をすべて分解
        $parts = explode('.', $filename);

        // 拡張子が1つだけならOK
        if (count($parts) <= 2) {
            return false;
        }
        return true;
    }
}
