<?php

namespace pictpostpersonal;

/**
 * 
 * CSRF対策に関する機能を提供するユーティリティクラス
 * 
 */
class CsrfUtility
{
    /**
     * 新しくCSRFトークンを発行する
     *
     * @return string 発行されたトークン
     * 
     */
    public static function publishCSRFToken(): string
    {
        Utility::SessionStart();
        $token_byte = random_bytes(20);
        $csrf_token = bin2hex($token_byte);
        $_SESSION['csrf_token'] = $csrf_token;
        return  $csrf_token;
    }

    /**
     * CSRFトークンをクリアする
     *
     * @return void
     * 
     */
    public static function clearCSRFToken(): void
    {
        unset($_SESSION['csrf_token']);
    }

    /**
     * CSRF トークンが有効かチェックする
     *
     * @param $csrf_token
     * 
     * @return bool
     * 
     */
    public static function checkCSRFToken(string $csrf_token): bool
    {
        Utility::SessionStart();
        if (isset($csrf_token)) {
            if (isset($_SESSION['csrf_token']) && hash_equals($_SESSION['csrf_token'], $csrf_token)) {
                return true;
            } else {
                return false;
            }
        } else {
            // CSRFトークンが存在しない場合
            return false;
        }
    }

    /**
     * CSRFトークン埋め込み用の<input type=hidden>HTMLコードを生成する
     *
     * @return string
     * 
     */
    public static function createCSRFTokenHtml(): string
    {
        Utility::SessionStart();
        if (!isset($_SESSION['csrf_token'])) {
            $csrf_token = self::publishCSRFToken();
        } else {
            $csrf_token = $_SESSION['csrf_token'];
        }
        $csrf_token = htmlspecialchars($csrf_token, ENT_QUOTES | ENT_HTML5, 'UTF-8');
        return "<input type=\"hidden\" name=\"csrf_token\" id=\"csrf_token\" value=\"$csrf_token\">";
    }
}
