<?php

namespace pictpostpersonal;

/**
 * 管理者ログイン関係を取り扱うクラス
 * 
 * ログイン情報は /env/usersetting.php に下記の変数名で定義されている
 *  $sys_username : ログインID
 *  $sys_password : パスワードのハッシュ値
 * 
 * メールアドレスによるパスワードリマインドやDB書き換えによらない簡単なID/パスワードリセットを実現するために
 * この構成となっている(usersetting.phpを削除すればID/パスワードをリセットできる)
 * 
 */
class Authentification
{
    //チェック状況をキャッシュする
    public static $check_token_cash = null;

    /**
     * ユーザーIDとパスワードを検証する
     *
     * @param string $user      ユーザーID
     * @param string $password  パスワード(平文)
     * 
     * @return bool 検証結果
     * 
     */
    public function login(string $user, string $password): bool
    {
        if (file_exists(USER_PATH)) {
            // /env/usersetting.phpを読み込み
            require_once(USER_PATH);

            if (LOCK_MODE) {
                return false; //LOCK_MODEが有効な場合はログイン不可
            }

            //SYS_USERNAMEおよびSYS_PASSWORDはusersetting.phpにて定義されている
            if (hash_equals(SYS_USERNAME, $user)  && password_verify($password, SYS_PASSWORD)) {
                $token_manager = new TokenManager(TOKEN_PATH);
                $token = $token_manager->generateToken('root'); //現状rootユーザーのみを想定

                // Cookieにトークンを保存
                $is_https = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off');
                $expiry =  time() + ($is_https ? 30 * 24 * 3600 : 3 * 3600); // HTTPS: 30日, HTTP: 3時間
                Utility::SetCookie('user-token', $token, $expiry, $is_https);
                return true;
            }
        }
        return false;
    }

    /**
     * ログアウトしログイントークンを削除する
     *
     * @return void
     * 
     */
    public function logout(): void
    {
        $user_token = Utility::GetCookie('user-token', '');
        if ($user_token === '') {
            return;
        }
        $token_manager = new TokenManager(TOKEN_PATH);
        $token_manager->revokeToken($user_token);
    }

    public function allLogout(): void
    {
        $token_manager = new TokenManager(TOKEN_PATH);
        $token_manager->revokeUserTokens('root'); //現状rootユーザーのみを想定
    }

    /**
     * ログイン済みかどうかチェックする
     *
     * @return bool ログイン済みか否か
     * 
     */
    public function isLogin(): bool
    {
        $user_token = Utility::GetCookie('user-token', '');
        if ($user_token === '') {
            return false;
        }
        $token_manager = new TokenManager(TOKEN_PATH);
        $token = $token_manager->validateToken($user_token);
        if ($token) {
            return true;
        } else {
            return false;
        }
    }
}
